<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Auth;
use App\Models\Message;
use App\Models\Conversation;
use App\Models\User;
use App\Models\Participant;
use DB;
use App\Events\MessageSent;
use App\Models\MessageRead;

class ChatController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $authUserId = Auth::id();
        $users = User::where('type', 'staff_member')
                        ->where('id', '!=', Auth::id())
                        ->get();
        $chatList = Conversation::whereHas('participants', function($q) use ($authUserId) {
                                                $q->where('user_id', $authUserId)->whereNull('left_at');
                                            })
                                    ->orderByDesc(
                                        Conversation::select('created_at')
                                            ->from('messages')
                                            ->whereColumn('messages.conversation_id', 'conversations.id')
                                            ->latest('created_at')
                                            ->limit(1)
                                    )
                                    ->get();
        return view('chats.list', [
            'chatList' => $chatList,
            'users' => $users
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $conversation = Conversation::find($id);
        $groupMemberIds = $conversation->participants->whereNull('left_at')->pluck('user_id');
        $receiver = $conversation->participants->where('user_id', '!=', Auth::id())->first()->user;
        $users = User::where('type', 'staff_member')
                        ->whereNotIn('id', $groupMemberIds)
                        ->get();
        $messageIds = $conversation->messages->where('sender_id', '!=', Auth::id())->pluck('id')->toArray();
        foreach ($messageIds as $msgId) {
            $message = MessageRead::where([
                'message_id' => $msgId,
                'user_id' => Auth::id(),
                'read_at' => null
            ])->first()
            ?->update(['read_at' => now()->addHours(5)]);
        }
        
        return view('chats.chat-box',[
            'conversation' => $conversation,
            'users' => $users,
            'receiver' => $receiver
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        if ($request->type == 'update_group_name') {
            $conversation = Conversation::find($id);
            $conversation->update(['name' => $request->group_name]);

            return redirect()->back()->with('success', 'Group chat name changed successfully');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, string $id)
    {
        if ($request->type == 'leave_group') {
            DB::beginTransaction();
            $messageObj = Message::create([
                                'conversation_id' => $id,
                                'sender_id' => Auth::id(), 
                                'message' => 'left the group',
                                'is_event' => true
                            ]);

            Participant::where([
                'conversation_id' => $id, 
                'user_id' => Auth::id(),
            ])->first()?->update(['left_at' => now()->addHours(5)]);
            
            $recipientIdArray = $messageObj->conversation->participants->where('user_id', '!=', Auth::id())->pluck('user_id')->toArray();
            // foreach ($recipientIdArray as $recipientId) {
            //     MessageRead::create([
            //                 'message_id' => $messageObj->id,
            //                 'user_id' => $recipientId
            //             ]);
            // }

            DB::commit();
            broadcast(new MessageSent($messageObj, $id, $recipientIdArray))->toOthers();
            
            return redirect()->route('chats.index')->with('success', 'You left the group');
        }
    }

    public function startConversation(Request $request){
        if ($request->type == 'add_member') {
            DB::beginTransaction();
            foreach ($request->members as $memberId) {
                $checkParticipant = Participant::where([
                                                    'conversation_id' => $request->conversation_id,
                                                    'user_id' => $memberId,
                                                ])->whereNotNull('left_at')->first();
                if ($checkParticipant) {
                    $checkParticipant->update([
                        'joined_at' => now()->addHours(5),
                        'left_at' => null
                    ]);
                }
                else{
                    Participant::firstOrCreate([
                                'conversation_id' => $request->conversation_id,
                                'user_id' => $memberId,
                                'joined_at' => now()->addHours(5)
                            ]);
                }
                $messageObj = Message::create([
                                'conversation_id' => $request->conversation_id,
                                'sender_id' => $memberId, 
                                'message' => 'joined the group',
                                'is_event' => true
                            ]);
                $recipientIdArray = $messageObj->conversation->participants->where('user_id', '!=', Auth::id())->pluck('user_id')->toArray();
                broadcast(new MessageSent($messageObj, $request->conversation_id, $recipientIdArray))->toOthers();
            }        
            DB::commit();
            
            return redirect()->back()->with('success', 'Member added successfully');
        }
        else{
            DB::beginTransaction();
            if (count($request->members) > 1) {
                $conversation = Conversation::firstOrCreate([
                                                'name' => 'Group Chat',
                                                'is_group' => true
                                            ]);
                $groupMemberID = $request->members;
                $groupMemberID[] = Auth::id();
                foreach($groupMemberID as $memberId){
                    Participant::firstOrCreate([
                                    'conversation_id' => $conversation->id,
                                    'user_id' => $memberId,
                                    'joined_at' => now()->addHours(5)
                                ]);
                }
            }
            else{
                $groupMemberID = $request->members;
                $memberID = $groupMemberID[0];
                $authUserId = Auth::id();

                $conversation = Conversation::where('is_group', false)
                                                    ->whereHas('participants', function ($q) use ($memberID) {
                                                        $q->where('user_id', $memberID);
                                                    })
                                                    ->whereHas('participants', function ($q) use ($authUserId) {
                                                        $q->where('user_id', $authUserId);
                                                    })
                                                    ->withCount('participants')
                                                    ->having('participants_count', 2)
                                                    ->first();
                if (!$conversation) {
                    $conversation = Conversation::create([
                                                'is_group' => false
                                            ]);
                    $groupMemberID[] = Auth::id();
                    foreach($groupMemberID as $memberId){
                        $participant = Participant::create([
                                        'conversation_id' => $conversation->id,
                                        'user_id' => $memberId,
                                        'joined_at' => now()->addHours(5)
                                    ]);
                    }
                }
            }
            DB::commit();
            return redirect()->route('chats.show', $conversation->id);
        }
    }

    public function chatBox($id){
        
    }

    public function sendMessage(Request $request){
        if($request->message){
            $messageObj = Message::create([
                                    'sender_id' => Auth::id(),
                                    'conversation_id' => $request->conversation_id,
                                    'message' => $request->message,
                                ]);
            $recipientIdArray = $messageObj->conversation->participants->where('user_id', '!=', Auth::id())->pluck('user_id')->toArray();
            foreach ($recipientIdArray as $recipientId) {
                MessageRead::create([
                            'message_id' => $messageObj->id,
                            'user_id' => $recipientId
                        ]);
            }
            broadcast(new MessageSent($messageObj, $request->conversation_id, $recipientIdArray))->toOthers();
        }


        return response()->json([
            'message' => $messageObj
        ]);
    }

    public function searchChat(Request $request){
        $search = $request->search;
        $authUserId = Auth::id();
        $users = User::where('type', 'staff_member')
                        ->where('id', '!=', Auth::id())
                        ->get();
        $chatList = Conversation::whereHas('participants', function($q) use ($authUserId) {
                                        $q->where('user_id', $authUserId)->whereNull('left_at');
                                    })
                                    ->orderByDesc(
                                        Conversation::select('created_at')
                                            ->from('messages')
                                            ->whereColumn('messages.conversation_id', 'conversations.id')
                                            ->latest('created_at')
                                            ->limit(1)
                                    )
                                    ->where(function ($q) use ($authUserId, $search) {
                                        $q->where(function ($q) use ($search) {
                                            $q->where('is_group', true)
                                            ->where('name', 'like', "%{$search}%");
                                        })
                                        ->orWhere(function ($q) use ($authUserId, $search) {
                                            $q->where('is_group', false)
                                            ->whereHas('participants', function ($q2) use ($authUserId, $search) {
                                                $q2->where('user_id', '!=', $authUserId)
                                                    ->whereHas('user', function ($q3) use ($search) {
                                                        $q3->where('name', 'like', "%{$search}%");
                                                    });
                                            });
                                        });
                                    })
                                    ->get();

        return view('chats.list', [
            'chatList' => $chatList,
            'users' => $users
        ]);
    }
}
